/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: Settings
    Created: 9 January, 2003
        $Id: Settings.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.settings;

import java.awt.Rectangle;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Properties;
import java.util.StringTokenizer;

import com.dgrossmann.photo.AppInfo;

/**
 * Represents the J Photo-Explorer program settings that are persisted using
 * the <tt>persist</tt> package.
 */
public class Settings
{
    // Default main image sizes.
    public static final int EXPDEFAULT_MAINIMG_LENGTH     = 600;
    public static final int EXPDEFAULT_MAINIMG_MAX_WIDTH  = 700;
    public static final int EXPDEFAULT_MAINIMG_MAX_HEIGHT = 450;
    public static final int EXPDEFAULT_MAINIMG_QUALITY    =  85;

    // Default index image sizes.
    public static final int EXPDEFAULT_INDEXIMG_LENGTH    = 200;
    public static final int EXPDEFAULT_INDEXIMG_QUALITY   =  75;

    // Default copyright text.
    public static final String DEFAULT_COPYRIGHT_STRING =
    	" ${year}";

    // Default XML encoding.
    public static final String METADEFAULT_XML_ENCODING= "ISO-8859-1";

    // Standard property names.
    public static final String GALLERY_TITLE           = "gallery.title";
    public static final String GALLERY_DESCRIPTION     = "gallery.description";
    public static final String GALLERY_COPYRIGHT       = "gallery.copyright";

    public static final String SERIES_CONTAINER_PREFIX = "series.container.";

    public static final String EXPORT_DIRECTORY          = "export.dir";
    public static final String EXPORT_MAINIMG_LENGTH     = "export.mainimg.length";
    public static final String EXPORT_MAINIMG_MAX_WIDTH  = "export.mainimg.max_width";
    public static final String EXPORT_MAINIMG_MAX_HEIGHT = "export.mainimg.max_height";
    public static final String EXPORT_MAINIMG_QUALITY    = "export.mainimg.quality";
    public static final String EXPORT_INDEXIMG_LENGTH    = "export.indeximg.length";
    public static final String EXPORT_INDEXIMG_QUALITY   = "export.indeximg.quality";
    public static final String EXPORT_USE_IMAGE_MAGICK   = "export.use_magick";

    public static final String METADATA_SAVE_XML       = "metadata.save.xml";
    public static final String METADATA_SAVE_ENCODING  = "metadata.save.xml.encoding";
    public static final String METADATA_EXPORT_XML     = "metadata.export.xml";
    public static final String METADATA_EXPORT_ENCODING= "metadata.export.xml.encoding";

    // Property data.
    private File              m_settingsFile;
    private Properties        m_properties;

    // Series container settings.
    private ArrayList<String> m_seriesContainerNames;

    /**
     * Creates a new <tt>Settings</tt> instance. The instance data is loaded
     * from the file specified in the <tt>connectString</tt> parameter.
     * @param settingsFileName - Name of the settings data file as a Java
     * properties file
     */
    public Settings (String settingsFileName)
    {
        m_settingsFile = new File(settingsFileName);
        if (!m_settingsFile.exists() && !m_settingsFile.isAbsolute())
        {
            m_settingsFile = new File(System.getProperty("user.home"),
                settingsFileName);
        }
        m_properties = new Properties();
        m_seriesContainerNames = new ArrayList<String>();
        // Set the default properties.
        this.setInt(EXPORT_USE_IMAGE_MAGICK, 1);
        // Initialize the properties.
        try
        {
            m_properties.load(new FileInputStream(m_settingsFile));
            // Extract and remove the series container directories.
            String name, dir;
            for (int i = 1;; i++)
            {
                name = SERIES_CONTAINER_PREFIX + i;
                dir = this.get(name);
                if (dir == null || dir.length() == 0)
                    break;
                m_properties.remove(name);
                this.addSeriesContainer(dir);
            }
        }
        catch (Exception ignored)
        {
            // If there is no properties file.
        }
    } // Settings

    /**
     * Gets an iterator for the names of the series container names.
     * @return Iterator for the series container names
     */
    public Iterator<String> getSeriesContainerNameIterator ()
    {
        return m_seriesContainerNames.iterator();
    } // getSeriesContainerNameIterator

    /**
     * Adds a series directory container. This is a directories containing
     * series directories.
     * @param containerDirName - Path of the directory to be added. Can be
     * absolute or relative to the current directory.
     */
    public void addSeriesContainer (String containerDirName)
    {
        if (!m_seriesContainerNames.contains(containerDirName))
            m_seriesContainerNames.add(containerDirName);
    } // addSeriesContainer

    /**
     * Gets a named property of this series structure instance.
     * @param name - Propery name
     * @return Property value as string
     */
    public String get (String name)
    {
        Object val = m_properties.getProperty(name);
        return (val != null) ? ((String) val) : "";
    } // get

    /**
     * Gets a named property of this series structure instance.
     * @param name - Propery name
     * @param defaultValue - Default value of property named <tt>name</tt> is
     * non-existent
     * @return Property value as string
     */
    public String get (String name, String defaultValue)
    {
        String val = this.get(name);
        return (val.length() > 0) ? val : defaultValue;
    } // get

    /**
     * Sets a named property of this series structure instance.
     * @param name - Propery name
     * @param val - New string value
     */
    public void set (String name, String val)
    {
        m_properties.setProperty(name, (val != null) ? val.trim() : "");
    } // set

    /**
     * Removes a named property from this series structure instance.
     * @param name - Property name
     */
    public void remove (String name)
    {
        m_properties.remove(name);
    } // remove

    /**
     * Gets a named property of this series structure instance.
     * @param name - Property name
     * @param defaultValue - Default value of property named <tt>name</tt> is
     * non-existent
     * @return Property value as int
     */
    public int getInt (String name, int defaultValue)
    {
        try
        {
            String val = m_properties.getProperty(name);
            if (val != null)
                return Integer.parseInt(val);
        }
        catch (Exception ignored)
        {
        }
        return defaultValue;
    } // getInt

    /**
     * Sets a named property of this series structure instance.
     * @param name - Property name
     * @param val - New integer value
     */
    public void setInt (String name, int val)
    {
        m_properties.setProperty(name, Integer.toString(val));
    } // setInt

    /**
     * Gets a named property of this series structure instance.
     * @param name - Property name
     * @param defaultValue - Default value of property named <tt>name</tt> is
     * non-existent
     * @return Property value as boolean
     */
    public boolean getBoolean (String name, boolean defaultValue)
    {
        try
        {
            String val = m_properties.getProperty(name);
            if (val != null && val.length() > 0)
            {
                char ch = Character.toLowerCase(val.charAt(0));
                return ch == 't' || ch == 'y' || ch == '1';
            }
        }
        catch (Exception ignored)
        {
        }
        return defaultValue;
    } // getBoolean

    /**
     * Sets a named property of this series structure instance.
     * @param name - Property name
     * @param val - New boolean value
     */
    public void setBoolean (String name, boolean val)
    {
        m_properties.setProperty(name, val ? "true" : "false");
    } // setBoolean

    /**
     * Gets a named property of this series structure instance.
     * @param name - Property name
     * @return Property value as Rectangle or <tt>null</tt> if non-existent
     */
    public Rectangle getRect (String name)
    {
        String valStr = m_properties.getProperty(name);
         if (valStr == null)
            return null;
        StringTokenizer tok = new StringTokenizer(valStr, ",");
        if (tok.countTokens() < 4)
            return null;
        try
        {
            Rectangle rect = new Rectangle();
            rect.x = Integer.parseInt(tok.nextToken());
            rect.y = Integer.parseInt(tok.nextToken());
            rect.width = Integer.parseInt(tok.nextToken());
            rect.height = Integer.parseInt(tok.nextToken());
            return rect;
        }
        catch (Exception exc)
        {
            return null;
        }
    } // getRect

    /**
     * Sets a named property of this series structure instance.
     * @param name - Property name
     * @param rect - New rectangle value
     */
    public void setRect (String name, Rectangle rect)
    {
        m_properties.setProperty(name, Integer.toString(rect.x) + "," + rect.y
            + "," + rect.width + "," + rect.height);
    } // setRect

    /**
     * Gets the property names used in this instance.
     * @return Property name array
     */
    public String[] getUsedPropertyNames ()
    {
        return m_properties.keySet().toArray(new String[0]);
    } // getUsedPropertyNames

    /**
     * Saves the state of this instance into the file whose name was given in
     * the constructor.
     */
    public void save ()
    {
        try
        {
            Properties props = new Properties();
            props.putAll(m_properties);
            // Add the series container directories.
            if (!m_seriesContainerNames.isEmpty())
            {
                int count = m_seriesContainerNames.size();
                for (int i = 1; i <= count; i++)
                {
                    props.setProperty(SERIES_CONTAINER_PREFIX + i,
                        m_seriesContainerNames.get(i-1).toString());
                }
            }
            // Remove properties with a null value.
            try
            {
                Iterator<Object> iter = props.keySet().iterator();
                String           name, val;
                while (iter.hasNext())
                {
                    name = iter.next().toString();
                    val = props.getProperty(name);
                    if (val == null || val.length() == 0)
                        props.remove(name);
                }
            }
            catch (Exception ignored)
            {
            }
            // Store the processed properties.
            props.store(new FileOutputStream(m_settingsFile),
                AppInfo.APP_NAME + " Settings");
        }
        catch (Exception exc)
        {
            exc.printStackTrace();
        }
    } // save
} // Settings
