/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ExportSelDialog
    Created: 9 January, 2003
        $Id: ExportSelDialog.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Iterator;

import javax.swing.JDialog;

import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.settings.Settings;

/**
 * The export selection dialog class.
 */
public class ExportSelDialog extends JDialog implements ItemListener
{
    private static final String SEP = "------------------------------";

    private SeriesContainer   m_seriesContainer;
    private Settings          m_settings;
    private boolean           m_bIsChanged;
    private DirectoryObject[] m_seriesDirs;
    private DirectoryObject   m_currentSeries, m_currentGroup;

    /**
     * Creates a new <tt>ExportSelDialog</tt> form instance.
     * @param parent - Parent frame
     * @param sContainer - Series container object that manages series
     * directories. There is exactly one instance per application.
     * @param settings - Settings object used to load and store the settings.
     * There is exactly one instance per application.
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public ExportSelDialog
       ( Frame           parent
       , SeriesContainer sContainer
       , Settings        settings
       , boolean         modal
       )
    {
        super(parent, modal);
        m_seriesContainer = sContainer;
        m_seriesDirs = m_seriesContainer.getSeriesDirectories();
        m_currentSeries = m_currentGroup = null;
        m_settings = settings;
        m_bIsChanged = false;
        this.initComponents();
        // Fill the export directory.
        exportDirLabel.setText(m_settings.get(Settings.EXPORT_DIRECTORY));
        // Fill the series combo box.
        seriesComboBox.addItem("All");
        seriesComboBox.addItem(SEP);
        for (int i = 0; i < m_seriesDirs.length; i++)
        {
            m_seriesContainer.ensureLoaded(m_seriesDirs[i]);
            seriesComboBox.addItem(m_seriesDirs[i].getTitlePlain());
        }
        seriesComboBox.addItemListener(this);
        // Fill the main group combo box.
        mainGroupComboBox.addItem("All");
        mainGroupComboBox.addItem(SEP);
        mainGroupComboBox.setEnabled(false);
        mainGroupComboBox.addItemListener(this);
        // Position centered in the parent dialog.
        Rectangle parentRect, thisRect;
        parentRect = parent.getBounds();
        thisRect = this.getBounds();
        thisRect.x = parentRect.x + (parentRect.width - thisRect.width) / 2;
        thisRect.y = parentRect.y + (parentRect.height - thisRect.height) / 2;
        if (thisRect.x < 0)
            thisRect.x = 0;
        if (thisRect.y < 0)
            thisRect.y = 0;
        this.setBounds(thisRect);
    } // ExportSelDialog

    /**
     * Shows the dialog window.
     * @return <tt>True</tt> if the user made changes, <tt>false</tt> if she
     * canceled the dialog
     */
    public boolean showDialog ()
    {
        this.setVisible(true);
        return m_bIsChanged;
    } // showDialog;

    /** Gets the series to export. */
    public DirectoryObject getSeries ()
    {
        return m_currentSeries;
    } // getSeries

    /** Gets the main group to export. */
    public DirectoryObject getGroup ()
    {
        return m_currentGroup;
    } // getGroup

    /** Gets whether images in the Main size should be exported. */
    public boolean getExportMain ()
    {
        return mainImgCheckBox.isSelected();
    } // getExportMain

    /** Gets whether images in the Index size should be exported. */
    public boolean getExportIndex ()
    {
        return indexImgCheckBox.isSelected();
    } // getExportIndex

    /** Gets whether to force image export. */
    public boolean getForceExport ()
    {
        return forceCheckBox.isSelected();
    } // getForceExport

    /* Selection change event listener method. */
    public void itemStateChanged (ItemEvent e)
    {
    	Iterator<DirectoryObject> subDirIter;
        int                       index;

        if (e.getSource() == seriesComboBox)
        {
            index = seriesComboBox.getSelectedIndex();
            index -= 2;
            m_currentSeries = m_currentGroup = null;
            mainGroupComboBox.removeAllItems();
            mainGroupComboBox.addItem("All");
            mainGroupComboBox.addItem(SEP);
            if (index < 0 || index >= m_seriesDirs.length)
            {
                mainGroupComboBox.setEnabled(false);
                return;
            }
            m_currentSeries = m_seriesDirs[index];
            mainGroupComboBox.setEnabled(true);
            subDirIter = m_currentSeries.getSubDirIterator();
            while (subDirIter.hasNext())
                mainGroupComboBox.addItem(subDirIter.next().getTitlePlain());
            return;
        }
        if (e.getSource() == mainGroupComboBox)
        {
            index = mainGroupComboBox.getSelectedIndex();
            index -= 2;
            m_currentGroup = null;
            if (m_currentSeries == null)
                return;
            m_currentGroup = m_currentSeries.getSubDirAt(index);
        }
    } // itemStateChanged

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        titleToLabel = new javax.swing.JLabel();
        exportDirLabel = new javax.swing.JLabel();
        titleNewerLabel = new javax.swing.JLabel();
        whichSeriesPanel = new javax.swing.JPanel();
        jLabel2 = new javax.swing.JLabel();
        jLabel3 = new javax.swing.JLabel();
        seriesComboBox = new javax.swing.JComboBox();
        jLabel4 = new javax.swing.JLabel();
        mainGroupComboBox = new javax.swing.JComboBox();
        whichImagesPanel = new javax.swing.JPanel();
        jLabel21 = new javax.swing.JLabel();
        mainImgCheckBox = new javax.swing.JCheckBox();
        indexImgCheckBox = new javax.swing.JCheckBox();
        forceCheckBox = new javax.swing.JCheckBox();
        buttonPanel = new javax.swing.JPanel();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        this.getContentPane().setLayout(new java.awt.GridBagLayout());

        this.setTitle("Export for the Web");
        this.setResizable(false);
        this.addWindowListener(new java.awt.event.WindowAdapter()
        {
            @Override
			public void windowClosing(java.awt.event.WindowEvent evt)
            {
                ExportSelDialog.this.closeDialog(evt);
            }
        });

        titleToLabel.setText("This dialog exports either all files or all files of a series to the Web export directory:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 6);
        this.getContentPane().add(titleToLabel, gridBagConstraints);

        exportDirLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        exportDirLabel.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 0, 6);
        this.getContentPane().add(exportDirLabel, gridBagConstraints);

        titleNewerLabel.setText("Only the files that are newer than those in the export directory are processed.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(3, 6, 6, 6);
        this.getContentPane().add(titleNewerLabel, gridBagConstraints);

        whichSeriesPanel.setLayout(new java.awt.GridBagLayout());

        whichSeriesPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(null, "Which Series"));
        jLabel2.setText("Specify whether all series, or only one series or group, are processed:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        whichSeriesPanel.add(jLabel2, gridBagConstraints);

        jLabel3.setText("Series:");
        jLabel3.setToolTipText("");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        whichSeriesPanel.add(jLabel3, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        whichSeriesPanel.add(seriesComboBox, gridBagConstraints);

        jLabel4.setText("Main Group:");
        jLabel4.setToolTipText("Main groups are direct subgroups of a series");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 12, 6, 0);
        whichSeriesPanel.add(jLabel4, gridBagConstraints);

        mainGroupComboBox.setToolTipText("Main groups are direct subgroups of a series");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        whichSeriesPanel.add(mainGroupComboBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        this.getContentPane().add(whichSeriesPanel, gridBagConstraints);

        whichImagesPanel.setLayout(new java.awt.GridBagLayout());

        whichImagesPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(null, "Which Image Size Types"));
        jLabel21.setText("Specify which image size types should be generated:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        whichImagesPanel.add(jLabel21, gridBagConstraints);

        mainImgCheckBox.setSelected(true);
        mainImgCheckBox.setText("Main Images (one per Web page)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 0);
        whichImagesPanel.add(mainImgCheckBox, gridBagConstraints);

        indexImgCheckBox.setSelected(true);
        indexImgCheckBox.setText("Index Images (small for overview pages)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(6, 12, 0, 0);
        whichImagesPanel.add(indexImgCheckBox, gridBagConstraints);

        forceCheckBox.setText("Always generate images (even if the exported images are newer)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 6, 6, 0);
        whichImagesPanel.add(forceCheckBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        this.getContentPane().add(whichImagesPanel, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ExportSelDialog.this.okButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(okButton);

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ExportSelDialog.this.cancelButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(cancelButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        this.getContentPane().add(buttonPanel, gridBagConstraints);

        this.pack();
    }// </editor-fold>//GEN-END:initComponents

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_cancelButtonActionPerformed
    {//GEN-HEADEREND:event_cancelButtonActionPerformed
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_cancelButtonActionPerformed

    private void okButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_okButtonActionPerformed
    {//GEN-HEADEREND:event_okButtonActionPerformed
        // Add your handling code here:
        m_bIsChanged = true;
        this.cancelButtonActionPerformed(evt);
    }//GEN-LAST:event_okButtonActionPerformed

    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeDialog

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JLabel exportDirLabel;
    private javax.swing.JCheckBox forceCheckBox;
    private javax.swing.JCheckBox indexImgCheckBox;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel21;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JComboBox mainGroupComboBox;
    private javax.swing.JCheckBox mainImgCheckBox;
    private javax.swing.JButton okButton;
    private javax.swing.JComboBox seriesComboBox;
    private javax.swing.JLabel titleNewerLabel;
    private javax.swing.JLabel titleToLabel;
    private javax.swing.JPanel whichImagesPanel;
    private javax.swing.JPanel whichSeriesPanel;
    // End of variables declaration//GEN-END:variables

} // ExportSelDialog
