/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ImageViewDialog
    Created: 20 February, 2006
        $Id: ExportedImageViewDialog.java 144 2008-03-02 08:42:32Z dirk $
  $Revision: 144 $
      $Date: 2008-03-02 09:42:32 +0100 (So, 02 Mrz 2008) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.*;
import java.io.File;

import javax.swing.ImageIcon;
import javax.swing.JDialog;
import javax.swing.JScrollBar;

import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.webexport.ExportFactory;

/**
 * The <i>image view</i> dialog class that shows an exported image.
 * @author Dirk Grossmann
 */
public class ExportedImageViewDialog extends JDialog
{
    private static final String EXPORTED_IMAGE_VIEW = "Exported Image View";

    private Settings        m_settings;
    private SeriesContainer m_seriesContainer;
    private Color           m_errorColor;
    private FileObject      m_fileObj;
    private Image           m_image;
    private ImageIcon       m_imageIcon;

    /**
     * Creates a new <tt>ExportedImageViewDialog form instance.
     * @param fileObj - File object representing the image to be shown
     * @param settings - The settings object (for getting the maximum image
     * sizes)
     * @param seriesContainer - The series container
     * @param parent - Parent dialog
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public ExportedImageViewDialog
        ( FileObject      fileObj
        , Settings        settings
        , SeriesContainer seriesContainer
        , Dialog          parent
        , boolean         modal
        )
    {
        super(parent, modal);
        this.initialize(fileObj, settings, seriesContainer);
    } // ExportedImageViewDialog

    /**
     * Creates a new <tt>ExportedImageViewDialog form instance.
     * @param fileObj - File object representing the image to be shown
     * @param settings - The settings object (for getting the maximum image
     * sizes)
     * @param seriesContainer - The series container
     * @param parent - Parent frame
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public ExportedImageViewDialog
        ( FileObject      fileObj
        , Settings        settings
        , SeriesContainer seriesContainer
        , Frame           parent
        , boolean         modal
        )
    {
        super(parent, modal);
        this.initialize(fileObj, settings, seriesContainer);
    } // ExportedImageViewDialog

    /**
     * Private method to initialize this dialog instance.
     * @param fileObj - File object representing the image to be shown
     * @param settings - The settings object (for getting the maximum image
     * sizes)
     * @param seriesContainer - The series container
     */
    private void initialize
        ( FileObject      fileObj
        , Settings        settings
        , SeriesContainer seriesContainer
        )
    {
        m_fileObj = fileObj;
        m_settings = settings;
        m_seriesContainer = seriesContainer;
        m_image = null;
        m_imageIcon = null;
        m_errorColor = new Color(153, 51, 51);
        this.initComponents();
        titleLabel.setFont(titleLabel.getFont().deriveFont(Font.BOLD));
        this.loadImage();
        // Position the dialog centered in the parent dialog.
        Rectangle parentRect, thisRect;
        parentRect = this.getParent().getBounds();
        thisRect = this.getBounds();
        thisRect.x = parentRect.x + (parentRect.width - thisRect.width) / 2;
        thisRect.y = parentRect.y + (parentRect.height - thisRect.height) / 2;
        if (thisRect.x < 0)
            thisRect.x = 0;
        if (thisRect.y < 0)
            thisRect.y = 0;
        this.setBounds(thisRect);
    } // initialize

    /**
     * Private method to load the exported main image from the file system
     * object member variable's value.
     */
    private void loadImage ()
    {
        File[] exportedFiles;
        File   f;
        String str;

        f = null;
        this.enableNavButtons();
        if (m_fileObj != null)
        {
            exportedFiles = ExportFactory.getExport(m_settings,
                m_seriesContainer,this.getParent()).getExportedFiles(m_fileObj);
            if (exportedFiles != null)
                f = exportedFiles[0];
        }
        if (f == null)
        {
            imageLabel.setIcon(null);
            imageLabel.setText("No exported image");
            titleLabel.setText(" ");
            sizeLabel.setText(" ");
            this.setTitle(EXPORTED_IMAGE_VIEW);
            return;
        }
        // Set the dialog title.
        str = m_fileObj.getTitlePlain();
        if (str.length() > 0)
            str += " - ";
        str += EXPORTED_IMAGE_VIEW;
        this.setTitle(str);
        titleLabel.setText(m_fileObj.getTitlePlain());
        // Try to load the image.
        this.getParent().setCursor(new Cursor(Cursor.WAIT_CURSOR));
        str = " ";
        try
        {
            Toolkit tk = Toolkit.getDefaultToolkit();
            MediaTracker track = new MediaTracker(this);
            m_image = tk.getImage(f.toURI().toURL());
            track.addImage(m_image, 0);
            try
            {
                track.waitForAll();
            }
            catch (InterruptedException iexc)
            {
            }
            m_imageIcon = new ImageIcon(m_image);
            // Set the image.
            if (m_imageIcon != null)
            {
                str = m_imageIcon.getIconWidth() + " x "
                    + m_imageIcon.getIconHeight() + " pixels  ("
                    + (f.length() / 1024L) + " KB)";
                imageLabel.setIcon(m_imageIcon);
                imageLabel.setPreferredSize(new Dimension(
                    m_imageIcon.getIconWidth(), m_imageIcon.getIconHeight()));
            }
        }
        catch (OutOfMemoryError me)
        {
            // Panic - no memory.
            m_image = null;
            m_imageIcon = null;
            Runtime.getRuntime().gc();
            str = "Out of memory while reading image - Please close the window.";
            sizeLabel.setForeground(m_errorColor);
        }
        catch (Exception e)
        {
            str = "Cannot load original image: " + e.getMessage();
            sizeLabel.setForeground(m_errorColor);
        }
        finally
        {
            this.getParent().setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
            sizeLabel.setText(str);
        }
    } // loadImage

    /**
     * Private method to enable or disable the navigation buttons.
     */
    private void enableNavButtons ()
    {
        boolean         bHavePrevious = false;
        boolean         bHaveNext     = false;
        DirectoryObject dirObj;
        FileObject      fileObj;
        int             index, i;

        if (m_fileObj != null)
        {
            // Check previous.
            dirObj = (DirectoryObject) m_fileObj.getParent();
            index = dirObj.getIndexOfFile(m_fileObj);
            for (i = index - 1; i >= 0; i--)
            {
                fileObj = dirObj.getFileAt(i);
                if (fileObj.isToExport() &&
                    fileObj.getFileType() == FileObject.TYPE_IMAGE_PREVIEW)
                {
                    bHavePrevious = true;
                    break;
                }
            }
            for (i = index + 1; i < dirObj.getFileCount(); i++)
            {
                fileObj = dirObj.getFileAt(i);
                if (fileObj.isToExport() &&
                    fileObj.getFileType() == FileObject.TYPE_IMAGE_PREVIEW)
                {
                    bHaveNext = true;
                    break;
                }
            }
        }
        prevImgButton.setEnabled(bHavePrevious);
        nextImgButton.setEnabled(bHaveNext);
    } // enableNavButtons

    /**
     * Gets the default preferred image size from the current settings.
     */
    private Dimension getDefaultSize ()
    {
        int prefWidth, maxWidth, prefHeight, sbWidth;

        prefWidth = m_settings.getInt(Settings.EXPORT_MAINIMG_LENGTH,
            Settings.EXPDEFAULT_MAINIMG_LENGTH);
        maxWidth = m_settings.getInt(Settings.EXPORT_MAINIMG_MAX_WIDTH,
            Settings.EXPDEFAULT_MAINIMG_MAX_WIDTH);
        prefHeight = m_settings.getInt(Settings.EXPORT_MAINIMG_MAX_HEIGHT,
            Settings.EXPDEFAULT_MAINIMG_MAX_HEIGHT);
        // Add the size of a scrollbar.
        JScrollBar tempSB = new JScrollBar();
        Dimension prefSBSize = tempSB.getUI().getPreferredSize(tempSB);
        sbWidth = (prefSBSize != null) ? prefSBSize.width : 0;
        return new Dimension(Math.max(maxWidth, prefWidth) + sbWidth,
            Math.max(prefWidth, prefHeight) + sbWidth);
    } // getDefaultSize

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        jScrollPane1 = new javax.swing.JScrollPane();
        imageLabel = new javax.swing.JLabel();
        titleLabel = new javax.swing.JLabel();
        sizeLabel = new javax.swing.JLabel();
        buttonPanel = new javax.swing.JPanel();
        prevImgButton = new javax.swing.JButton();
        nextImgButton = new javax.swing.JButton();
        sepLabel = new javax.swing.JLabel();
        closeButton = new javax.swing.JButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        addWindowListener(new java.awt.event.WindowAdapter()
        {
            public void windowClosing(java.awt.event.WindowEvent evt)
            {
                closeDialog(evt);
            }
        });

        imageLabel.setMaximumSize(this.getDefaultSize());
        imageLabel.setPreferredSize(this.getDefaultSize());
        jScrollPane1.setViewportView(imageLabel);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        getContentPane().add(jScrollPane1, gridBagConstraints);

        titleLabel.setForeground(new java.awt.Color(0, 70, 213));
        titleLabel.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        getContentPane().add(titleLabel, gridBagConstraints);

        sizeLabel.setText("3000 x 3000 pixels");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 0);
        getContentPane().add(sizeLabel, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        prevImgButton.setText("< Previous");
        prevImgButton.setToolTipText("Go to the previous exported image");
        prevImgButton.setMaximumSize(new java.awt.Dimension(110, 28));
        prevImgButton.setMinimumSize(new java.awt.Dimension(110, 28));
        prevImgButton.setPreferredSize(new java.awt.Dimension(110, 28));
        prevImgButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                prevImgButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(prevImgButton);

        nextImgButton.setText("Next >");
        nextImgButton.setToolTipText("Go to the next exported image");
        nextImgButton.setMaximumSize(new java.awt.Dimension(110, 28));
        nextImgButton.setMinimumSize(new java.awt.Dimension(110, 28));
        nextImgButton.setPreferredSize(new java.awt.Dimension(110, 28));
        nextImgButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                nextImgButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(nextImgButton);

        sepLabel.setText(" ");
        buttonPanel.add(sepLabel);

        closeButton.setText("Close");
        closeButton.setMaximumSize(new java.awt.Dimension(110, 28));
        closeButton.setMinimumSize(new java.awt.Dimension(110, 28));
        closeButton.setPreferredSize(new java.awt.Dimension(110, 28));
        closeButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                closeButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(closeButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        getContentPane().add(buttonPanel, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void nextImgButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_nextImgButtonActionPerformed
    {//GEN-HEADEREND:event_nextImgButtonActionPerformed
        DirectoryObject dirObj;
        FileObject      fileObj;
        int             index, i;

        if (m_fileObj == null)
            return;
        dirObj = (DirectoryObject) m_fileObj.getParent();
        index = dirObj.getIndexOfFile(m_fileObj);
        for (i = index + 1; i < dirObj.getFileCount(); i++)
        {
            fileObj = dirObj.getFileAt(i);
            if (fileObj.isToExport() &&
                fileObj.getFileType() == FileObject.TYPE_IMAGE_PREVIEW)
            {
                m_fileObj = fileObj;
                this.loadImage();
                break;
            }
        }
    }//GEN-LAST:event_nextImgButtonActionPerformed

    private void prevImgButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_prevImgButtonActionPerformed
    {//GEN-HEADEREND:event_prevImgButtonActionPerformed
        DirectoryObject dirObj;
        FileObject      fileObj;
        int             index, i;

        if (m_fileObj == null)
            return;
        dirObj = (DirectoryObject) m_fileObj.getParent();
        index = dirObj.getIndexOfFile(m_fileObj);
        for (i = index - 1; i >= 0; i--)
        {
            fileObj = dirObj.getFileAt(i);
            if (fileObj.isToExport() &&
                fileObj.getFileType() == FileObject.TYPE_IMAGE_PREVIEW)
            {
                m_fileObj = fileObj;
                this.loadImage();
                break;
            }
        }
    }//GEN-LAST:event_prevImgButtonActionPerformed

    private void closeButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_closeButtonActionPerformed
    {//GEN-HEADEREND:event_closeButtonActionPerformed
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeButtonActionPerformed
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeDialog
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton closeButton;
    private javax.swing.JLabel imageLabel;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JButton nextImgButton;
    private javax.swing.JButton prevImgButton;
    private javax.swing.JLabel sepLabel;
    private javax.swing.JLabel sizeLabel;
    private javax.swing.JLabel titleLabel;
    // End of variables declaration//GEN-END:variables

} // ExportedImageViewDialog
