/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: PropertiesDialog
    Created: 2 January, 2003
        $Id: PropertiesDialog.java 53 2006-02-11 10:43:09Z Dirk $
  $Revision: 53 $
      $Date: 2006-02-11 11:43:09 +0100 (Sa, 11 Feb 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JTextField;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.ui.ExplorerMainFrame;
import com.dgrossmann.photo.ui.ImageHolder;
import com.dgrossmann.photo.ui.ShellExec;

/**
 * Implements the dialog class for showing the <i>properties</i> of
 * directories, files, and references.
 * @author Dirk Grossmann
 */
public class PropertiesDialog extends JDialog implements ItemListener
{    
    private ExplorerMainFrame m_parent;
    private ImageHolder       m_imgHolder;
    private AbstractFSObject  m_fileObj;
    private String            m_oldFileName;
    private String            m_beginDate, m_endDate;
    private int               m_exportQuality;
    private boolean           m_isChanged;

    // Additional widgets based on file type.
    private JLabel            endDateLabel;
    private JTextField        endDateText;
    private JButton           setEndDateButton;
    private JLabel            hrefLabel;
    private JTextField        hrefText;

    /**
     * Creates a new <tt>PropertiesDialog</tt> form instance.
     * @param imgHolder - Image holder object
     * @param fileObj - File object whose properties are to be displayed
     * @param exportedFiles - Array of 2 elements representing the exported
     * files (<tt>null</tt> if not present). It is used to display the file
     * sizes.
     * @param parent - Parent main frame in the user interface
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public PropertiesDialog
        ( ImageHolder       imgHolder
        , AbstractFSObject  fileObj
        , File[]            exportedFiles
        , ExplorerMainFrame parent
        , boolean           modal
        )
    {
        super(parent, modal);
        m_parent = parent;
        // Fill the instance variables.
        m_imgHolder = imgHolder;
        m_fileObj = fileObj;
        m_isChanged = false;
        if (m_fileObj == null)
            return;
        m_beginDate = fileObj.get(AbstractFSObject.DATE_BEGIN);
        m_endDate = fileObj.get(AbstractFSObject.DATE_END);
        endDateLabel = hrefLabel = null;
        endDateText = hrefText = null;
        setEndDateButton = null;
        parent.setCursor(new Cursor(Cursor.WAIT_CURSOR));
        // Initialize the UI components.
        this.initComponents();
        // Set the dialog title.
        String tit = "";
        String tend = "";
        if (m_fileObj.getParent() == null)
        {
            tit = "Series \"";
            tend = "\"";
            descLabel.setText("Series description:");
        }
        else if (m_fileObj instanceof DirectoryObject)
        {
            tit = "Group \"";
            tend = "\"";
            descLabel.setText("Group description:");
        }
        else
            descLabel.setText("Description:");
        tit += m_fileObj.getTitlePlain();
        if (tit.length() > 0)
            tit += tend + " - ";
        this.setTitle(tit + "Properties");
        // Append the remaining widgets for directories and references.
        int currentRow = 6;
        GridBagConstraints gridBagConstraints;
        if (m_fileObj instanceof DirectoryObject)
        {
            // End date label.
            if (m_fileObj.getParent() == null)
                tit = " date of this series";
            else
                tit = " date of this group";
            endDateLabel = new JLabel();
            endDateLabel.setText("End Date: ");
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = currentRow;
            gridBagConstraints.insets = new Insets(4, 0, 0, 0);
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            dataPanel.add(endDateLabel, gridBagConstraints);
            // End date text.
            endDateText = new JTextField();
            endDateText.setEditable(false);
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 1;
            gridBagConstraints.gridy = currentRow;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.insets = new Insets(4, 0, 0, 0);
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.weightx = 100.0;
            dataPanel.add(endDateText, gridBagConstraints);
            // Set end date button.
            setEndDateButton = new JButton();
            setEndDateButton.setText("...");
            setEndDateButton.setToolTipText
                ("Select the end" + tit);
            setEndDateButton.setMaximumSize(new Dimension(30, 20));
            setEndDateButton.setMinimumSize(new Dimension(30, 20));
            setEndDateButton.setPreferredSize(new Dimension(30, 20));
            setEndDateButton.addActionListener(new ActionListener()
            {
                public void actionPerformed(java.awt.event.ActionEvent evt)
                {
                    setEndDateButtonActionPerformed(evt);
                }
            });
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 2;
            gridBagConstraints.gridy = currentRow;
            gridBagConstraints.insets = new Insets(4, 4, 0, 0);
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            dataPanel.add(setEndDateButton, gridBagConstraints);
            // Set the specific properties.
            nameLabel.setText("Name: ");
            beginDateLabel.setText("Begin Date: ");
            setBeginDateButton.setToolTipText
                ("Select the begin" + tit);
            endDateText.setText(fileObj.expandDate(m_endDate, false));
        }
        if (m_fileObj.isReference())
        {
            // Referenced document label.
            hrefLabel = new JLabel();
            hrefLabel.setText("Referenced Document URL:");
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = currentRow;
            gridBagConstraints.gridwidth = 3;
            gridBagConstraints.insets = new Insets(6, 0, 0, 0);
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            dataPanel.add(hrefLabel, gridBagConstraints);
            currentRow++;
            // Referenced document text.
            hrefText = new JTextField();
            hrefText.setToolTipText
                ("URL of the document that should be referenced");
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = currentRow;
            gridBagConstraints.gridwidth = 3;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.weightx = 100.0;
            dataPanel.add(hrefText, gridBagConstraints);
            // Set the specific properties.
            nameText.setEnabled(false);
            nameText.setEditable(false);
            tit = m_fileObj.getFileNamePart();
            if (tit.length() == 0)
                tit = m_fileObj.getFileName();
            if (tit.length() > 0 &&
                !tit.equalsIgnoreCase(AbstractFSObject.REFERENCE))
            {
                nameText.setText("(" + tit + ")");
            }
            hrefText.setText(m_fileObj.get(AbstractFSObject.HREF));
        }
        // Set the properties from the current file.
        m_oldFileName = m_fileObj.getFileName();
        nameText.setText(m_oldFileName);
        openButton.setEnabled((m_fileObj instanceof FileObject) &&
            !m_fileObj.isReference());
        String moddate = m_fileObj.getModDateTimeString(false);
        if (!m_fileObj.isReference())
        {
	        String sz = m_fileObj.getFileSizeStr(true);
	        if (sz != null)
	            moddate += "    Size: " + sz;
        }
        else
            modDateLabel.setText("");
        modDateValueLabel.setText(moddate);
        titleText.setText(AbstractFSObject.transformAccents(
            m_fileObj.getTitle(true), false));
        titleText.requestFocus();
        subTitleText.setText(AbstractFSObject.transformAccents(
            m_fileObj.get(AbstractFSObject.SUBTITLE), false));
        locationText.setText(AbstractFSObject.transformAccents(
            m_fileObj.get(AbstractFSObject.LOCATION), false));
        beginDateText.setText(fileObj.expandDate(m_beginDate, false));
        descTextArea.setText(AbstractFSObject.transformAccents(
            m_fileObj.get(AbstractFSObject.DESCRIPTION), false));
        descTextArea.select(0, 0);
        toExportCheckBox.setSelected(m_fileObj.isToExport());
        toExportCheckBox.addItemListener(this);
        qualityText.setEnabled(m_fileObj.isToExport());
        m_exportQuality = m_fileObj.getConversionQuality();
        if (m_fileObj.isToExport() && m_exportQuality > 0)
            qualityText.setText(Integer.toString(m_exportQuality));
        else
        {
            qualityText.setText("Default");
            m_exportQuality = 0;
        }
        mainSizeTextLabel.setText("---");
        indexSizeTextLabel.setText("---");
        if (m_fileObj instanceof FileObject && exportedFiles != null)
        {
            long len;
            if (exportedFiles[0] != null)
            {
                len = exportedFiles[0].length() / 1024;
                if (len <= 0)
                    len = 1;
                mainSizeTextLabel.setText(Long.toString(len) + " KB");
            }
            if (exportedFiles[1] != null)
            {
                len = exportedFiles[1].length() / 1024;
                if (len <= 0)
                    len = 1;
                indexSizeTextLabel.setText(Long.toString(len) + " KB");
            }
        }
        // Position the dialog centered in the parent dialog.
        Rectangle parentRect, thisRect;
        parentRect = parent.getBounds();
        thisRect = this.getBounds();
        thisRect.x = parentRect.x + (parentRect.width - thisRect.width) / 2;
        thisRect.y = parentRect.y + (parentRect.height - thisRect.height) / 2;
        if (thisRect.x < 0)
            thisRect.x = 0;
        if (thisRect.y < 0)
            thisRect.y = 0;
        this.setBounds(thisRect);
        // Set the preview image.
        m_imgHolder.addTransientScaleCategory(ImageHolder.SC_PROPERTIES);
        Dimension pSize = previewPanel.getSize();
        pSize.width = pSize.height;
        previewPanel.setSize(pSize);
        ImageIcon previewImg = m_imgHolder.getTypeImage(m_fileObj, true,
            ImageHolder.SC_PROPERTIES, pSize.height, this, true);
        if (previewImg != null)
        {
            previewImgLabel.setIcon(previewImg);
            if (m_fileObj instanceof FileObject &&
                ((FileObject) m_fileObj).getFileType() ==
                    FileObject.TYPE_IMAGE_PREVIEW)
            {
                previewImgLabel.setToolTipText(null);
            }
        }
        String str = m_imgHolder.getLastError();
        if (str.length() > 0)
        {
            JOptionPane.showMessageDialog(this, str, AppInfo.APP_NAME,
                JOptionPane.ERROR_MESSAGE);
            previewImgLabel.setText("Preview not available");
        }
        if ((m_fileObj instanceof FileObject) &&
            ((FileObject) m_fileObj).getFileType() == FileObject.TYPE_IMAGE)
        {
            previewImgLabel.setText("Preview not available");
        }
        parent.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    } // PropertiesDialog

    /**
     * Shows the dialog window.
     * @return <tt>True</tt> if the user made changes, <tt>false</tt> if she
     * canceled the dialog
     */
    public boolean showDialog ()
    {
        this.setVisible(true);
        return m_isChanged;
    } // showDialog

    /**
     * Private method to get the heading for the Properties border.
     * @return The heading string
     */
    private String getPropertiesHeading ()
    {
        if (m_fileObj instanceof DirectoryObject)
            return "Folder Properties";
        return "File Properties";
    } // getPropertiesHeading

    /**
     * State change listener method.
     * @param e - The item event
     * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
     */
    public void itemStateChanged (ItemEvent e)
    {
        if (e.getSource() == toExportCheckBox)
        {
            if (toExportCheckBox.isSelected())
            {
                qualityText.setEnabled(true);
                if (m_exportQuality == 0)
                    qualityText.setText("Default");
                else
                    qualityText.setText(Integer.toString(m_exportQuality));
            }
            else
            {
                try
                {
                    m_exportQuality = Integer.parseInt
                        (qualityText.getText().trim());
                }
                catch (Exception exc)
                {
                    m_exportQuality = 0;
                }
                qualityText.setEnabled(false);
                qualityText.setText("Default");
            }
            return;
        }
    } // itemStateChanged

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        propertiesPanel = new javax.swing.JPanel();
        previewPanel = new javax.swing.JPanel();
        previewImgLabel = new javax.swing.JLabel();
        dataPanel = new javax.swing.JPanel();
        nameLabel = new javax.swing.JLabel();
        nameText = new javax.swing.JTextField();
        openButton = new javax.swing.JButton();
        modDateLabel = new javax.swing.JLabel();
        modDateValueLabel = new javax.swing.JLabel();
        titleLabel = new javax.swing.JLabel();
        titleText = new javax.swing.JTextField();
        subTitleLabel = new javax.swing.JLabel();
        subTitleText = new javax.swing.JTextField();
        locationLabel = new javax.swing.JLabel();
        locationText = new javax.swing.JTextField();
        beginDateLabel = new javax.swing.JLabel();
        beginDateText = new javax.swing.JTextField();
        setBeginDateButton = new javax.swing.JButton();
        descPanel = new javax.swing.JPanel();
        descLabel = new javax.swing.JLabel();
        descScrollPane = new javax.swing.JScrollPane();
        descTextArea = new javax.swing.JTextArea();
        exportPanel = new javax.swing.JPanel();
        toExportCheckBox = new javax.swing.JCheckBox();
        qualityLabel = new javax.swing.JLabel();
        qualityText = new javax.swing.JTextField();
        percentLabel = new javax.swing.JLabel();
        fileSsizePanel = new javax.swing.JPanel();
        mainSizeLabel = new javax.swing.JLabel();
        mainSizeTextLabel = new javax.swing.JLabel();
        indexSizeLabel = new javax.swing.JLabel();
        indexSizeTextLabel = new javax.swing.JLabel();
        buttonPanel = new javax.swing.JPanel();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setTitle("Properties");
        addWindowListener(new java.awt.event.WindowAdapter()
        {
            public void windowClosing(java.awt.event.WindowEvent evt)
            {
                closeDialog(evt);
            }
        });

        propertiesPanel.setLayout(new java.awt.GridBagLayout());

        propertiesPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(null, this.getPropertiesHeading()));
        previewPanel.setLayout(new java.awt.BorderLayout());

        previewPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(""));
        previewPanel.setMinimumSize(new java.awt.Dimension(170, 170));
        previewPanel.setPreferredSize(new java.awt.Dimension(170, 170));
        previewImgLabel.setForeground(new java.awt.Color(102, 102, 102));
        previewImgLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        previewImgLabel.setToolTipText("Preview image (if available)");
        previewImgLabel.setMaximumSize(new java.awt.Dimension(2147483647, 2147483647));
        previewImgLabel.setMinimumSize(new java.awt.Dimension(100, 100));
        previewImgLabel.setPreferredSize(new java.awt.Dimension(160, 160));
        previewPanel.add(previewImgLabel, java.awt.BorderLayout.CENTER);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        propertiesPanel.add(previewPanel, gridBagConstraints);

        dataPanel.setLayout(new java.awt.GridBagLayout());

        nameLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        nameLabel.setText("File Name:  ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        dataPanel.add(nameLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        dataPanel.add(nameText, gridBagConstraints);

        openButton.setText("Open");
        openButton.setMaximumSize(new java.awt.Dimension(72, 22));
        openButton.setMinimumSize(new java.awt.Dimension(72, 22));
        openButton.setPreferredSize(new java.awt.Dimension(72, 22));
        openButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                openButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        dataPanel.add(openButton, gridBagConstraints);

        modDateLabel.setText("Last Modified:  ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(modDateLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(modDateValueLabel, gridBagConstraints);

        titleLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        titleLabel.setText("Title:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 0, 0, 0);
        dataPanel.add(titleLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 0, 0, 0);
        dataPanel.add(titleText, gridBagConstraints);

        subTitleLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        subTitleLabel.setText("Subtitle:  ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(subTitleLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(subTitleText, gridBagConstraints);

        locationLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        locationLabel.setText("Location:  ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(locationLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(locationText, gridBagConstraints);

        beginDateLabel.setText("Date: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(beginDateLabel, gridBagConstraints);

        beginDateText.setEditable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 0, 0, 0);
        dataPanel.add(beginDateText, gridBagConstraints);

        setBeginDateButton.setText("...");
        setBeginDateButton.setToolTipText("Select the date of this image");
        setBeginDateButton.setMaximumSize(new java.awt.Dimension(30, 20));
        setBeginDateButton.setMinimumSize(new java.awt.Dimension(30, 20));
        setBeginDateButton.setPreferredSize(new java.awt.Dimension(30, 20));
        setBeginDateButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                setBeginDateButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 0);
        dataPanel.add(setBeginDateButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        propertiesPanel.add(dataPanel, gridBagConstraints);

        descPanel.setLayout(new java.awt.BorderLayout());

        descLabel.setText("Description:");
        descPanel.add(descLabel, java.awt.BorderLayout.NORTH);

        descTextArea.setColumns(80);
        descTextArea.setFont(new java.awt.Font("Monospaced", 0, 12));
        descTextArea.setLineWrap(true);
        descTextArea.setRows(10);
        descTextArea.setWrapStyleWord(true);
        descScrollPane.setViewportView(descTextArea);

        descPanel.add(descScrollPane, java.awt.BorderLayout.CENTER);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        propertiesPanel.add(descPanel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        getContentPane().add(propertiesPanel, gridBagConstraints);

        exportPanel.setLayout(new java.awt.GridBagLayout());

        exportPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(null, "Export for the Web"));
        toExportCheckBox.setText("Export to Web Directory");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        exportPanel.add(toExportCheckBox, gridBagConstraints);

        qualityLabel.setText("JPEG Quality:");
        qualityLabel.setToolTipText("Quality for the JPEG conversion (0 - 100%)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 12, 0, 0);
        exportPanel.add(qualityLabel, gridBagConstraints);

        qualityText.setText("Default");
        qualityText.setToolTipText("Quality for the JPEG conversion (0 - 100%)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 3, 0, 0);
        exportPanel.add(qualityText, gridBagConstraints);

        percentLabel.setText("%");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 3, 0, 4);
        exportPanel.add(percentLabel, gridBagConstraints);

        fileSsizePanel.setLayout(new java.awt.GridBagLayout());

        mainSizeLabel.setText("Main Image Size:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        fileSsizePanel.add(mainSizeLabel, gridBagConstraints);

        mainSizeTextLabel.setText("3333 KB");
        mainSizeTextLabel.setPreferredSize(new java.awt.Dimension(60, 16));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        fileSsizePanel.add(mainSizeTextLabel, gridBagConstraints);

        indexSizeLabel.setText("Index Image Size:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 12, 6, 0);
        fileSsizePanel.add(indexSizeLabel, gridBagConstraints);

        indexSizeTextLabel.setText("3333 KB");
        indexSizeTextLabel.setPreferredSize(new java.awt.Dimension(60, 16));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 0);
        fileSsizePanel.add(indexSizeTextLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        exportPanel.add(fileSsizePanel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        getContentPane().add(exportPanel, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                okButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(okButton);

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                cancelButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(cancelButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 6, 6);
        getContentPane().add(buttonPanel, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void setEndDateButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_setEndDateButtonActionPerformed
    {//GEN-HEADEREND:event_setEndDateButtonActionPerformed
        DateChooserDialog dlg = new DateChooserDialog
			(this, m_endDate, "end", true);
        if (!dlg.showDialog())
            return;
        m_endDate = dlg.getDateString();
        if (endDateText != null)
            endDateText.setText(m_fileObj.expandDate(m_endDate, false));
    }//GEN-LAST:event_setEndDateButtonActionPerformed

    private void setBeginDateButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_setBeginDateButtonActionPerformed
    {//GEN-HEADEREND:event_setBeginDateButtonActionPerformed
        String dlgTitle;
        
        if (m_fileObj instanceof DirectoryObject)
            dlgTitle = "begin";
        else
            dlgTitle = "";
        DateChooserDialog dlg = new DateChooserDialog
			(this, m_beginDate, dlgTitle, true);
        if (!dlg.showDialog())
            return;
        m_beginDate = dlg.getDateString();
        beginDateText.setText(m_fileObj.expandDate(m_beginDate, false));
    }//GEN-LAST:event_setBeginDateButtonActionPerformed

    private void openButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_openButtonActionPerformed
    {//GEN-HEADEREND:event_openButtonActionPerformed
        if (m_fileObj.isReference() || !(m_fileObj instanceof FileObject))
            return;
        ShellExec exec = new ShellExec(this);
        exec.openDocument((FileObject) m_fileObj);
    }//GEN-LAST:event_openButtonActionPerformed

    private void okButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_okButtonActionPerformed
    {//GEN-HEADEREND:event_okButtonActionPerformed
        String  s;
        
        // Set and verify properties.
        if (toExportCheckBox.isSelected())
        {
            try
            {
                m_exportQuality = Integer.parseInt
                    (qualityText.getText().trim());
            }
            catch (Exception e)
            {
                m_exportQuality = 0;
            }
            if (m_exportQuality < 0 || m_exportQuality > 100)
            {
                JOptionPane.showMessageDialog(this, "The JPEG conversion " +
                    "quality must be between 0 and 100", AppInfo.APP_NAME,
                    JOptionPane.WARNING_MESSAGE);
                return;
            }
            m_fileObj.setToExport(true);
            m_fileObj.setConversionQuality(m_exportQuality);
        }
        else
        {
            m_fileObj.setToExport(false);
            m_fileObj.setConversionQuality(0);
        }
        // Set properties.
        if (!m_fileObj.isReference())
        {
            s = nameText.getText();
            if (s != null && s.length() > 0 &&
                !s.equalsIgnoreCase(m_oldFileName))
            {
                // Does a file of the desired name already exist ?-
                if (m_fileObj.getParent() != null)
                {
                    File f = new File(m_fileObj.getParent().getFileName(), s);
                    if (f.exists())
                    {
                        JOptionPane.showMessageDialog(m_parent,
                            "Cannot rename as a file with the following "
                            + "name already exists:\n" + f.getAbsolutePath(),
                            AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                        return;
                    }
                }
                // No, continue with renaming.
                m_imgHolder.renameThumbnails(m_fileObj, s);
                m_parent.renameExportedFiles(m_fileObj, null, s);
                m_fileObj.setFileName(s, true);
            }
        }
        s = titleText.getText();
        if (s != null)
            m_fileObj.set(AbstractFSObject.TITLE, s.trim());
        s = subTitleText.getText();
        if (s != null)
            m_fileObj.set(AbstractFSObject.SUBTITLE, s.trim());
        if (m_fileObj.isReference())
        {
            s = hrefText.getText();
            if (s != null)
                m_fileObj.set(AbstractFSObject.HREF, s.trim());
        }
        s = locationText.getText();
        if (s != null)
            m_fileObj.set(AbstractFSObject.LOCATION, s.trim());
        s = descTextArea.getText();
        if (s != null)
            m_fileObj.set(AbstractFSObject.DESCRIPTION, s.trim());
        m_fileObj.set(AbstractFSObject.DATE_BEGIN, m_beginDate);
        if (m_fileObj instanceof DirectoryObject)
            m_fileObj.set(AbstractFSObject.DATE_END, m_endDate);
        // Mark the containing directory as changed.
        if (m_fileObj instanceof FileObject)
        {
            DirectoryObject parentDir = (DirectoryObject) m_fileObj.getParent();
            if (parentDir != null)
                parentDir.setChanged();
        }
        else if (m_fileObj instanceof DirectoryObject)
            ((DirectoryObject) m_fileObj).setChanged();
        // Close the dialog.
        m_isChanged = true;
        this.cancelButtonActionPerformed(evt);
    }//GEN-LAST:event_okButtonActionPerformed

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_cancelButtonActionPerformed
    {//GEN-HEADEREND:event_cancelButtonActionPerformed
        this.setVisible(false);
        this.dispose();
        Runtime.getRuntime().gc();
    }//GEN-LAST:event_cancelButtonActionPerformed
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
        Runtime.getRuntime().gc();
    }//GEN-LAST:event_closeDialog
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel beginDateLabel;
    private javax.swing.JTextField beginDateText;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JPanel dataPanel;
    private javax.swing.JLabel descLabel;
    private javax.swing.JPanel descPanel;
    private javax.swing.JScrollPane descScrollPane;
    private javax.swing.JTextArea descTextArea;
    private javax.swing.JPanel exportPanel;
    private javax.swing.JPanel fileSsizePanel;
    private javax.swing.JLabel indexSizeLabel;
    private javax.swing.JLabel indexSizeTextLabel;
    private javax.swing.JLabel locationLabel;
    private javax.swing.JTextField locationText;
    private javax.swing.JLabel mainSizeLabel;
    private javax.swing.JLabel mainSizeTextLabel;
    private javax.swing.JLabel modDateLabel;
    private javax.swing.JLabel modDateValueLabel;
    private javax.swing.JLabel nameLabel;
    private javax.swing.JTextField nameText;
    private javax.swing.JButton okButton;
    private javax.swing.JButton openButton;
    private javax.swing.JLabel percentLabel;
    private javax.swing.JLabel previewImgLabel;
    private javax.swing.JPanel previewPanel;
    private javax.swing.JPanel propertiesPanel;
    private javax.swing.JLabel qualityLabel;
    private javax.swing.JTextField qualityText;
    private javax.swing.JButton setBeginDateButton;
    private javax.swing.JLabel subTitleLabel;
    private javax.swing.JTextField subTitleText;
    private javax.swing.JLabel titleLabel;
    private javax.swing.JTextField titleText;
    private javax.swing.JCheckBox toExportCheckBox;
    // End of variables declaration//GEN-END:variables
    
} // PropertiesDialog
