/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: PropertyPanel.java
    Created: 16 December, 2005
        $Id: PropertyPanel.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.io.File;
import java.util.EventObject;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.ui.ExplorerMainFrame;
import com.dgrossmann.photo.ui.ImageHolder;
import com.dgrossmann.photo.ui.dialog.DateChooserDialog;
import com.dgrossmann.photo.ui.dialog.PropertiesDialog;
import com.dgrossmann.photo.webexport.ExportException;
import com.dgrossmann.photo.webexport.ExportFactory;
import com.dgrossmann.photo.webexport.IWebExport;

/**
 * Panel class for the Properties Overview panel.
 * @author Dirk
 */
public class PropertyPanel extends AbstractExplorerPanel
    implements FocusListener, PanelTitleChangeListener
{
    private static final String PROPS_OVERVIEW  = "Properties";
    private static final String PROPS_SERIES    = "Series";
    private static final String PROPS_DIR       = "Group";
    private static final String PROPS_FILE      = "File";
    private static final String PROPS_REFERENCE = "Reference";
    private static final String PROPS_SEPARATOR = "Separator";

    private ImageHolder      m_imgHolder;
    private AbstractFSObject m_currentFSObject;
    private boolean          m_bIsPreviewShowing;
    private File[]           m_exportedFiles;

    /**
     * Creates a new <tt>PropertyPanel</tt> instance.
     * @param frm - The Explorer frame
     */
    public PropertyPanel (ExplorerMainFrame frm)
    {
        super(frm, PROPS_OVERVIEW);
        m_currentFSObject = null;
        m_imgHolder = frm.getImageHolder();
        m_exportedFiles = null;
        m_bIsPreviewShowing = false;
        this.initComponents();
        this.add(this.getTitleLabel(), BorderLayout.NORTH);
        // Add us as focus listener.
        titleText.addFocusListener(this);
        descTextArea.addFocusListener(this);
        // Set the current object.
        this.setCurrentFSObjects(null);
    } // PropertyPanel

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#setupComponents()
     */
    public void setupComponents ()
    {
    } // setupComponents

    /**
     * @see com.dgrossmann.photo.ui.panel.AbstractExplorerPanel#setActiveColors()
     */
    @Override
	public void setActiveColors ()
    {
        if (this.getCurrentFSObjects() != null &&
            this.getCurrentFSObjects().size() > 0)
        {
            super.setActiveColors();
        }
        else
            super.setInactiveColors();
    } // setActiveColors

    /**
     * @see com.dgrossmann.photo.ui.panel.AbstractExplorerPanel#setCurrentFSObjects(java.util.List)
     */
    @Override
	public void setCurrentFSObjects (List<AbstractFSObject> currentFsObjects)
    {
        // Is this a superfluous call?
        if (currentFsObjects != null &&
            currentFsObjects.equals(this.getCurrentFSObjects()))
        {
            return;
        }
        // The user could have changed the properties of the old object (the
        // corresponding Focus-Lost event will follow later).
        this.setAttributesFromControls(null);
        // If not, proceed.
        super.setCurrentFSObjects(currentFsObjects);
        // If we have less than 2 object, we set the current FS object.
        if (currentFsObjects == null || currentFsObjects.size() != 1)
            m_currentFSObject = null;
        else
            m_currentFSObject = currentFsObjects.get(0);
        this.refresh();
    } // setCurrentFSObjects

    /**
     * Private method to get the current file system object if we have
     * exactly one.
     * @return The current file system object
     */
    private AbstractFSObject getCurrentFSObject ()
    {
        return m_currentFSObject;
    } // getCurrentFSObject

    /**
     * @see java.awt.event.FocusListener#focusGained(java.awt.event.FocusEvent)
     */
    public void focusGained (FocusEvent e)
    {
        // On entering a component, we do not have to do anything.
    } // focusGained

    /**
     * @see java.awt.event.FocusListener#focusLost(java.awt.event.FocusEvent)
     */
    public void focusLost (FocusEvent e)
    {
        this.setAttributesFromControls(e);
        this.firePropertiesChanged(m_currentFSObject);
    } // focusLost

    /**
     * Private method to set the attributes of the current file system object
     * from the values in the UI controls of this panel.
     * @param e - An AWT event, may be <tt>null</tt>
     */
    private void setAttributesFromControls (EventObject e)
    {
        String val;

        if (m_currentFSObject == null)
            return;
        // Set the values in the current file system object.
        if (e == null || e.getSource() == titleText)
        {
            val = titleText.getText();
            if (val != null &&
                !val.equals(m_currentFSObject.get(AbstractFSObject.TITLE)))
            {
                m_currentFSObject.set(AbstractFSObject.TITLE, val.trim());
            }
        }
        if (e == null || e.getSource() == descTextArea)
        {
            val = descTextArea.getText();
            if (val != null &&
                !val.equals(m_currentFSObject.get(AbstractFSObject.DESCRIPTION)))
            {
                m_currentFSObject.set(AbstractFSObject.DESCRIPTION, val.trim());
            }
        }
        // Mark the containing directory as changed.
        if (m_currentFSObject instanceof FileObject)
        {
            DirectoryObject parentDir = (DirectoryObject)
                m_currentFSObject.getParent();
            if (parentDir != null)
                parentDir.setChanged();
        }
        else if (m_currentFSObject instanceof DirectoryObject)
            ((DirectoryObject) m_currentFSObject).setChanged();
    } // setAttributesFromControls

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#saveChanges()
     */
    public void saveChanges ()
    {
    } // saveChanges

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#refresh()
     */
    public void refresh ()
    {
        mainSizeTextLabel.setText("--");
        indexSizeTextLabel.setText("--");
        m_exportedFiles = null;
        if (m_currentFSObject == null)
        {
            // Set the default texts and disable all controls.
            this.disableControls(true);
            this.setInactiveColors();
            // Set the title.
            List<AbstractFSObject> allFSObjs = this.getCurrentFSObjects();
            if (allFSObjs == null || allFSObjs.size() == 0 ||
                allFSObjs.size() == 1 && allFSObjs.get(0) == null)
            {
                this.setTitle(PROPS_OVERVIEW);
                this.setInactiveColors();
            }
            else
            {
                String title = Integer.toString(allFSObjs.size())
                    + " objects selected";
                this.setTitle(title);
                this.setActiveColors();
            }
        }
        else
        {
            this.setActiveColors();
            String prefix = "";
            if (m_currentFSObject instanceof DirectoryObject)
            {
                if (((DirectoryObject) m_currentFSObject).getParent() == null)
                    prefix = PROPS_SERIES;
                else
                    prefix = PROPS_DIR;
            }
            else if (m_currentFSObject instanceof FileObject)
            {
                FileObject fileObj = (FileObject) m_currentFSObject;
                if (fileObj.isSeparator())
                {
                    this.setTitle(PROPS_SEPARATOR);
                    this.disableControls(true);
                    this.fillPreview();
                    return;
                }
                else if (fileObj.isReference())
                    prefix = PROPS_REFERENCE;
                else
                    prefix = PROPS_FILE;
            }
            this.enableControls();
            // Set the file system object properties.
            titleText.setText(AbstractFSObject.transformAccents(
                m_currentFSObject.getTitle(true), false));
            beginDateText.setText(m_currentFSObject.getBeginDate(true));
            descTextArea.setText(AbstractFSObject.transformAccents(
                m_currentFSObject.get(AbstractFSObject.DESCRIPTION), false));
            descTextArea.select(0, 0);
            // Set the title.
            String panelTitle = m_currentFSObject.getTitlePlain();
            if (panelTitle.length() < 30)
                 panelTitle = prefix + " \"" + panelTitle + "\"";
            if ((m_currentFSObject instanceof DirectoryObject) &&
                panelTitle.length() < 25)
            {
                String title = m_currentFSObject.getTitlePlain();
                String dirName = m_currentFSObject.getFileName();
                if (title != null && !title.equals(dirName))
                    panelTitle += " (\"" + dirName + "\")";
            }
            this.setTitle(panelTitle);
            // Set the file sizes.
            if (m_currentFSObject instanceof FileObject)
            {
                m_exportedFiles = ExportFactory.getExport(
                    this.getFrame().getSettings(),
                    this.getFrame().getSeriesContainer(), this).
                    getExportedFiles(m_currentFSObject);
                if (m_exportedFiles != null)
                {
                    long len;
                    if (m_exportedFiles[0] != null)
                    {
                        len = m_exportedFiles[0].length() / 1024;
                        if (len <= 0)
                            len = 1;
                        mainSizeTextLabel.setText(Long.toString(len) + " KB");
                    }
                    if (m_exportedFiles[1] != null)
                    {
                        len = m_exportedFiles[1].length() / 1024;
                        if (len <= 0)
                            len = 1;
                        indexSizeTextLabel.setText(Long.toString(len) + " KB");
                    }
                }
            }
        }
        this.fillPreview();
    } // refresh

    /**
     * @see com.dgrossmann.photo.ui.panel.PanelTitleChangeListener#onTitleSelectionChanged(int)
     */
    public void onTitleSelectionChanged (int panelId)
    {
        m_bIsPreviewShowing = (panelId == PanelTitleChangeListener.SEL_PREVIEW);
        this.fillPreview();
    } // stateChanged

    /**
     * Private method to fill the preview pane.
     */
    private void fillPreview ()
    {
        if (m_currentFSObject == null || !m_bIsPreviewShowing ||
            ((m_currentFSObject instanceof FileObject) &&
            ((FileObject) m_currentFSObject).isSeparator()))
        {
            previewImgLabel.setText(null);
            previewImgLabel.setIcon(null);
            return;
        }
        // Set the preview image.
        this.getFrame().setCursor(new Cursor(Cursor.WAIT_CURSOR));
        m_imgHolder.addTransientScaleCategory(ImageHolder.SC_PROPERTIES);
        Dimension pSize = previewPanel.getSize();
        pSize.width = pSize.height;
        previewPanel.setSize(pSize);
        ImageIcon previewImg = m_imgHolder.getTypeImage(m_currentFSObject, true,
            ImageHolder.SC_PROPERTIES, pSize.height, this, true);
        if (previewImg != null)
        {
            previewImgLabel.setIcon(previewImg);
            if (m_currentFSObject instanceof FileObject &&
                ((FileObject) m_currentFSObject).getFileType() ==
                    FileObject.TYPE_IMAGE_PREVIEW)
            {
                previewImgLabel.setToolTipText(null);
            }
        }
        previewImgLabel.setText(null);
        String str = m_imgHolder.getLastError();
        if (str.length() > 0)
        {
            JOptionPane.showMessageDialog(this, str, AppInfo.APP_NAME,
                JOptionPane.ERROR_MESSAGE);
            previewImgLabel.setText("Preview not available");
        }
        if ((m_currentFSObject instanceof FileObject) &&
            ((FileObject) m_currentFSObject).getFileType()
            == FileObject.TYPE_IMAGE)
        {
            previewImgLabel.setText("Preview not available");
        }
        this.getFrame().setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    } // fillPreview

    /**
     * Disables all controls in the panel.
     * @param bSetDefaultTexts - <tt>True</tt> to set all texts to their
     * default empty values
     */
    private void disableControls (boolean bSetDefaultTexts)
    {
        titleText.setEnabled(false);
        titleText.setEditable(false);
        descTextArea.setEnabled(false);
        descTextArea.setEditable(false);
        propsButton.setEnabled(false);
        setBeginDateButton.setEnabled(false);
        if (bSetDefaultTexts)
        {
            titleText.setText("");
            beginDateText.setText("");
            descTextArea.setText("");
        }
    } // disableControls

    /**
     * Enables all controls in the panel.
     */
    private void enableControls ()
    {
        titleText.setEnabled(true);
        titleText.setEditable(true);
        descTextArea.setEnabled(true);
        descTextArea.setEditable(true);
        propsButton.setEnabled(true);
        setBeginDateButton.setEnabled(true);
    } // enableControls

    // Property names for the settings of this pane.
    private static final String PROP_SHOW_PREVIEW = "property_pane.show_preview";

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#loadSettings(com.dgrossmann.photo.settings.Settings)
     */
    public void loadSettings (Settings settings)
    {
        // Set the previously selected tab.
        m_bIsPreviewShowing = settings.getBoolean(PROP_SHOW_PREVIEW, false);
        this.getTitleLabel().setPanelComponents(aspectsPanel, propertyPanel,
            previewPanel, m_bIsPreviewShowing);
        this.getTitleLabel().addChangeListener(this);
    } // loadSettings

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#saveSettings(com.dgrossmann.photo.settings.Settings)
     */
    public void saveSettings (Settings settings)
    {
        // Save which tab is selected.
        settings.setBoolean(PROP_SHOW_PREVIEW, m_bIsPreviewShowing);
    } // saveSettings

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents()
    {
        java.awt.GridBagConstraints gridBagConstraints;

        aspectsPanel = new javax.swing.JPanel();
        propertyPanel = new javax.swing.JPanel();
        titleLabel = new javax.swing.JLabel();
        titleText = new javax.swing.JTextField();
        propsButton = new javax.swing.JButton();
        beginDateLabel = new javax.swing.JLabel();
        beginDateText = new javax.swing.JTextField();
        setBeginDateButton = new javax.swing.JButton();
        descScrollPane = new javax.swing.JScrollPane();
        descTextArea = new javax.swing.JTextArea();
        previewPanel = new javax.swing.JPanel();
        previewCenterPanel = new javax.swing.JPanel();
        previewImgLabel = new javax.swing.JLabel();
        fileSizePanel = new javax.swing.JPanel();
        mainSizeLabel = new javax.swing.JLabel();
        mainSizeTextLabel = new javax.swing.JLabel();
        indexSizeLabel = new javax.swing.JLabel();
        indexSizeTextLabel = new javax.swing.JLabel();

        this.setLayout(new java.awt.BorderLayout());

        aspectsPanel.setLayout(new java.awt.CardLayout());

        propertyPanel.setLayout(new java.awt.GridBagLayout());

        titleLabel.setText("Title:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 0);
        propertyPanel.add(titleLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 6, 0, 4);
        propertyPanel.add(titleText, gridBagConstraints);

        propsButton.setText("Properties");
        propsButton.setToolTipText("Displays the properties dialog");
        propsButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                PropertyPanel.this.propsButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 6, 0, 4);
        propertyPanel.add(propsButton, gridBagConstraints);

        beginDateLabel.setText("Date:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(6, 4, 0, 0);
        propertyPanel.add(beginDateLabel, gridBagConstraints);

        beginDateText.setEditable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 2);
        propertyPanel.add(beginDateText, gridBagConstraints);

        setBeginDateButton.setText(" ... ");
        setBeginDateButton.setToolTipText("Sets the begin date");
        setBeginDateButton.setMaximumSize(new java.awt.Dimension(30, 20));
        setBeginDateButton.setMinimumSize(new java.awt.Dimension(30, 20));
        setBeginDateButton.setPreferredSize(new java.awt.Dimension(30, 20));
        setBeginDateButton.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                PropertyPanel.this.setBeginDateButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 2, 0, 0);
        propertyPanel.add(setBeginDateButton, gridBagConstraints);

        descTextArea.setColumns(20);
        descTextArea.setLineWrap(true);
        descTextArea.setRows(3);
        descTextArea.setWrapStyleWord(true);
        descScrollPane.setViewportView(descTextArea);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        propertyPanel.add(descScrollPane, gridBagConstraints);

        aspectsPanel.add(propertyPanel, "normal");

        previewPanel.setLayout(new java.awt.GridBagLayout());

        previewCenterPanel.setLayout(new java.awt.BorderLayout());

        previewCenterPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(""));
        previewImgLabel.setForeground(new java.awt.Color(102, 102, 102));
        previewImgLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        previewImgLabel.setToolTipText("Preview image (if available)");
        previewImgLabel.setMaximumSize(new java.awt.Dimension(2147483647, 2147483647));
        previewImgLabel.setMinimumSize(new java.awt.Dimension(100, 100));
        previewImgLabel.setPreferredSize(new java.awt.Dimension(160, 160));
        previewCenterPanel.add(previewImgLabel, java.awt.BorderLayout.CENTER);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 6, 6);
        previewPanel.add(previewCenterPanel, gridBagConstraints);

        aspectsPanel.add(previewPanel, "preview");

        this.add(aspectsPanel, java.awt.BorderLayout.CENTER);

        fileSizePanel.setLayout(new java.awt.GridBagLayout());

        mainSizeLabel.setText("Main Image:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 0);
        fileSizePanel.add(mainSizeLabel, gridBagConstraints);

        mainSizeTextLabel.setText("--");
        mainSizeTextLabel.setPreferredSize(new java.awt.Dimension(60, 16));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 0);
        fileSizePanel.add(mainSizeTextLabel, gridBagConstraints);

        indexSizeLabel.setText("Index Image:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(2, 8, 2, 0);
        fileSizePanel.add(indexSizeLabel, gridBagConstraints);

        indexSizeTextLabel.setText("--");
        indexSizeTextLabel.setPreferredSize(new java.awt.Dimension(60, 16));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 0);
        fileSizePanel.add(indexSizeTextLabel, gridBagConstraints);

        this.add(fileSizePanel, java.awt.BorderLayout.SOUTH);

    }// </editor-fold>//GEN-END:initComponents

    private void setBeginDateButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_setBeginDateButtonActionPerformed
    {//GEN-HEADEREND:event_setBeginDateButtonActionPerformed
        AbstractFSObject fsObj = this.getCurrentFSObject();
        if (fsObj == null)
            return;
        DateChooserDialog dlg = new DateChooserDialog (this.getFrame(),
            fsObj.get(AbstractFSObject.DATE_BEGIN), "begin", true);
        if (!dlg.showDialog ())
            return;
        fsObj.set(AbstractFSObject.DATE_BEGIN, dlg.getDateString ());
        beginDateText.setText(fsObj.getBeginDate(true));
        if (fsObj instanceof DirectoryObject)
            ((DirectoryObject) fsObj).setChanged();
        this.firePropertiesChanged(fsObj);
    }//GEN-LAST:event_setBeginDateButtonActionPerformed

    private void propsButtonActionPerformed (java.awt.event.ActionEvent evt)//GEN-FIRST:event_propsButtonActionPerformed
    {//GEN-HEADEREND:event_propsButtonActionPerformed
        AbstractFSObject fsObj = this.getCurrentFSObject();
        if (fsObj == null)
            return;
        this.getFrame().save();
        boolean oldToExport = fsObj.isToExport();
        int oldQuality = fsObj.getConversionQuality();
        PropertiesDialog dlg = new PropertiesDialog(m_imgHolder,
            fsObj, m_exportedFiles, this.getFrame(), true);
        if (!dlg.showDialog())
            return;
        // Handle re-export of the selected file.
        if (fsObj instanceof FileObject &&
            !fsObj.isToExport())
        {
            try
            {
                ExportFactory.getExport(this.getFrame().getSettings(),
                    this.getFrame().getSeriesContainer(), this).
                    deleteExportedFiles(fsObj, IWebExport.EXPORT_ALL);
            }
            catch (ExportException ignored)
            {
            }
        }
        else if (fsObj instanceof FileObject &&
             ((!oldToExport && fsObj.isToExport()) ||
              (oldQuality != fsObj.getConversionQuality())))
        {
            this.getFrame().startExport(fsObj, true, true, true);
        }
        // Fire the refresh event after export.
        this.refresh();
        this.firePropertiesChanged(fsObj);
    }//GEN-LAST:event_propsButtonActionPerformed


    // Variables declaration - do not modify//GEN-BEGIN:variables
    public javax.swing.JPanel aspectsPanel;
    public javax.swing.JLabel beginDateLabel;
    public javax.swing.JTextField beginDateText;
    public javax.swing.JScrollPane descScrollPane;
    public javax.swing.JTextArea descTextArea;
    public javax.swing.JPanel fileSizePanel;
    public javax.swing.JLabel indexSizeLabel;
    public javax.swing.JLabel indexSizeTextLabel;
    public javax.swing.JLabel mainSizeLabel;
    public javax.swing.JLabel mainSizeTextLabel;
    public javax.swing.JPanel previewCenterPanel;
    public javax.swing.JLabel previewImgLabel;
    public javax.swing.JPanel previewPanel;
    public javax.swing.JPanel propertyPanel;
    public javax.swing.JButton propsButton;
    public javax.swing.JButton setBeginDateButton;
    public javax.swing.JLabel titleLabel;
    public javax.swing.JTextField titleText;
    // End of variables declaration//GEN-END:variables
} // PropertyPanel
