/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: BaseExporter
    Created: 9 January, 2003
        $Id: BaseExporter.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.webexport;

import java.awt.Component;
import java.awt.Dimension;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.settings.Settings;

/**
 * Base class of all Web exporters.
 */
public abstract class BaseExporter implements IWebExport
{
    protected Settings        m_settings;
    protected SeriesContainer m_seriesContainer;
    protected Component       m_parent;

    /**
     * Constructor that initializes the members of the abstract
     * <tt>BaseExporter</tt> class.
     */
    public BaseExporter ()
    {
        m_settings = null;
        m_seriesContainer = null;
        m_parent = null;
    } // BaseExporter

    /**
     * Sets the important objects.
     * @param settings - Settings object
     * @param sContainer - Series container object
     * @param parent - Parent component
     */
    public void setEnvironment
        ( Settings        settings
        , SeriesContainer sContainer
        , Component       parent
        )
    {
        m_settings = settings;
        m_seriesContainer = sContainer;
        m_parent = parent;
    } // setEnvironment

    /**
     * Protected method to get the copyright string from a file object.
     * @param fileObj - The image file object
     * @return The copyright string or <tt>null</tt> to not print a copyright
     */
    protected String getCopyrightString (FileObject fileObj)
    {
    	String  copyrightStr;
    	Integer year;
    	int     i;

    	copyrightStr = m_settings.get(Settings.GALLERY_COPYRIGHT);
    	if (copyrightStr == null || copyrightStr.trim().length() == 0)
    		return null;
    	i = copyrightStr.indexOf("${year}");
    	if (i > 0)
    	{
    		year = fileObj.getYear();
    		copyrightStr = copyrightStr.substring(0, i) + ((year != null) ?
   				year.toString() : "") + copyrightStr.substring(i + 7);
    	}
    	return copyrightStr;
    } // getCopyrightString

    /**
     * Private helper to get the new image size.
     * @param origSize - Original size
     * @param bIsMain - <tt>True</tt> for main images
     * @return The new size
     */
    private Dimension getNewSize (Dimension origSize, boolean bIsMain)
    {
        Dimension newSize = new Dimension();
        int       prefLength, maxWidth, maxHeight, newWidth, newHeight;

        if (!bIsMain)
        {
        	// For index images, we just use the length setting.
            prefLength = m_settings.getInt(Settings.EXPORT_INDEXIMG_LENGTH,
                Settings.EXPDEFAULT_INDEXIMG_LENGTH);
        	if (Math.max(origSize.width, origSize.height) <= prefLength)
        	{
        		newSize.width = origSize.width;
        		newSize.height = origSize.height;
        	}
            else if (origSize.width >= origSize.height)
            {
        		newHeight = (origSize.height * prefLength) / origSize.width;
        		newSize.width = prefLength;
        		newSize.height = newHeight;
            }
            else
            {
        		newWidth = (origSize.width * prefLength) / origSize.height;
        		newSize.width = newWidth;
        		newSize.height = prefLength;
            }
            return newSize;
        }
        // For main images, it is more complicated.
        prefLength = m_settings.getInt(Settings.EXPORT_MAINIMG_LENGTH,
            Settings.EXPDEFAULT_MAINIMG_LENGTH);
        maxWidth = m_settings.getInt(Settings.EXPORT_MAINIMG_MAX_WIDTH,
            Settings.EXPDEFAULT_MAINIMG_MAX_WIDTH);
        maxHeight = m_settings.getInt(Settings.EXPORT_MAINIMG_MAX_HEIGHT,
            Settings.EXPDEFAULT_MAINIMG_MAX_HEIGHT);
    	if (Math.max(origSize.width, origSize.height) <= prefLength)
    	{
    		newSize.width = origSize.width;
    		newSize.height = origSize.height;
    	}
        else if (origSize.height + (origSize.height/10) >= origSize.width)
        {
        	// Portrait or (almost) quadratic image. For portraits, the maximum
        	// height setting may be lower than the length setting.
        	if (prefLength > maxHeight)
        		prefLength = maxHeight;
    		newWidth = (origSize.width * prefLength) / origSize.height;
    		newSize.width = newWidth;
    		newSize.height = prefLength;
    		if (newSize.width < (newSize.height / 2) &&
    			prefLength < maxHeight)
    		{
    			// Small portraits are enlarged provided that the max. height
    			// setting is greater than the length.
	    		newWidth = (origSize.width * maxHeight) / origSize.height;
	    		newSize.width = newWidth;
	    		newSize.height = maxHeight;
    		}
        }
        else
        {
        	// Landscape image.
    		newHeight = (origSize.height * prefLength) / origSize.width;
    		newSize.width = prefLength;
    		newSize.height = newHeight;
    		if (newSize.height < (newSize.width / 2))
    		{
	    		newHeight = (origSize.height * maxWidth) / origSize.width;
	    		newSize.width = maxWidth;
	    		newSize.height = newHeight;
    		}
        }
        return newSize;
    } // getNewSize

    /**
	 * Protected method to fill the progress object with the sizes for the small
	 * images, provided that the original image size has been set.
	 * @param progress - Export progress object
	 */
    protected void fillImageSizes (ExportProgress progress)
    {
        Dimension origSize = progress.getOriginalSize();

        if (origSize == null)
            return;
        // Fill the main image size;
        if (!progress.isMainUpToDate())
            progress.setMainSize(this.getNewSize(origSize, true));
        if (!progress.isIndexUpToDate())
            progress.setIndexSize(this.getNewSize(origSize, false));
    } // fillImageSizes

    /**
     * Gets the conversion quality for the JPEG conversion.
     * @param fileObj - File system object to be converted
     * @param bMain - <tt>True</tt> for main images, <tt>false</tt> for index
     * (overview) images
     * @return Conversion quality in the range 1-100
     */
    public int getConversionQuality (AbstractFSObject fileObj, boolean bMain)
    {
        AbstractFSObject fsObj   = fileObj;
        int              quality = 0;

        while (fsObj != null && quality == 0)
        {
            quality = fsObj.getConversionQuality();
            fsObj = fsObj.getParent();
        }
        if (quality > 0)
            return quality;
        if (bMain)
        {
            return m_settings.getInt(Settings.EXPORT_MAINIMG_QUALITY,
                Settings.EXPDEFAULT_MAINIMG_QUALITY);
        }
        return m_settings.getInt(Settings.EXPORT_INDEXIMG_QUALITY,
            Settings.EXPDEFAULT_INDEXIMG_QUALITY);
    } // getConversionQuality

    /**
     * Protected template method that really carries out the file export. The
     * <tt>progress</tt> object is filled with the output directory, the file
     * names to be generated and whether to generate them (up-to-date setting);
     * and the output directory exists.
     * @param fileObj - File object to export
     * @param progress - Filled progress object
     * @throws ExportException on error
     */
    protected abstract void exportFile
        ( FileObject     fileObj
        , ExportProgress progress
        )
        throws ExportException;

    /**
	 * Private helper for the "export" method to ensure that the output
	 * directory exists. Returns the output directory object.
	 * @param progress - The progress object
	 * @return The output directory
	 * @throws ExportException
	 */
    private File prepareOutputDir
        ( ExportProgress progress
        ) throws ExportException
    {
        FileObject       fileObj = progress.getFileObject();
        List<String>     parents;
        Iterator<String> iter;
        DirectoryObject  seriesDir;
        boolean          bOK;
        File             dir;

        parents = new ArrayList<String>();
        seriesDir = (DirectoryObject) fileObj.getParent();
        if (seriesDir == null)
        {
            throw new ExportException(fileObj,
                "The file does not have a parent directory object");
        }
        while (seriesDir.getParent() != null)
        {
            parents.add(0, seriesDir.getFileName().toLowerCase());
            seriesDir = (DirectoryObject) seriesDir.getParent();
        }
        if (!m_seriesContainer.isSeriesDirectory(seriesDir))
        {
            throw new ExportException(fileObj,
                "The file does not have a valid series directory: "
                + seriesDir.getTitle(true));
        }
        dir = new File(m_settings.get(Settings.EXPORT_DIRECTORY));
        if (!dir.isDirectory())
        {
            throw new ExportException(fileObj, dir.toString()
                + " is not a valid Web export directory");
        }
        bOK = true;
        dir = new File(dir, seriesDir.getFileName().toLowerCase());
        if (!dir.isDirectory())
            bOK = dir.mkdir();
        iter = parents.iterator();
        while (bOK && iter.hasNext())
        {
            dir = new File(dir, iter.next());
            if (bOK && !dir.isDirectory())
                bOK = dir.mkdir();
        }
        if (!bOK)
        {
            throw new ExportException(fileObj,
                "Cannot create directory: " + dir);
        }
        progress.setOutputDirectory(dir.getAbsolutePath());
        return dir;
    } // prepareOutputDir

    /**
     * Exports a series, group, or image to the Web export directory.
     * @param progress - Export progress holder object that holds the file
     * object to export and is used to display export status information
     * @param whatToExport - Can be one of <tt>EXPORT_ALL</tt>, export all image
     * forms; <tt>EXPORT_MAIN_IMAGE</tt>, export only the main image; or
     * <tt>EXPORT_OVERVIEW_IMAGE</tt>, export only the overview image.
     * @param bForce - <tt>True</tt> to force export even if the exported files
     * already exist and are younger than the originals
     * @throws ExportException on failure
     */
    public void export
        ( ExportProgress progress
        , int            whatToExport
        , boolean        bForce
        )
        throws ExportException
    {
        FileObject fileObj = progress.getFileObject();
        File       dir, destFile;
        String     str, name;
        long       origMod, expMod;

        // Ensure that the output directory exists.
        dir = this.prepareOutputDir(progress);
        // Prepare the progrss object.
        str = fileObj.getBaseFileName().toLowerCase();
        name = str + ".jpg";
        progress.setMainFileName(name);
        origMod = (new File(fileObj.getFullPath())).lastModified();
        destFile = new File(dir, name);
        if (destFile.exists())
            expMod = destFile.lastModified();
        else
            expMod = 0;
        if (bForce || origMod > expMod)
            progress.setMainUpToDate(false);
        else
            progress.setMainUpToDate(true);
        name = str + "-s.jpg";
        progress.setIndexFileName(name);
        destFile = new File(dir, name);
        if (destFile.exists())
            expMod = destFile.lastModified();
        else
            expMod = 0;
        if (bForce || origMod > expMod)
            progress.setIndexUpToDate(false);
        else
            progress.setIndexUpToDate(true);
        // Honor the "what to export" flag.
        if (whatToExport == IWebExport.EXPORT_NOTHING)
            return;
        if (whatToExport == IWebExport.EXPORT_MAIN_IMAGE)
            progress.setIndexUpToDate(true);
        else if (whatToExport == IWebExport.EXPORT_OVERVIEW_IMAGE)
            progress.setMainUpToDate(true);
        // Call the export template method.
        if (!progress.isMainUpToDate() || !progress.isIndexUpToDate())
        {
            this.exportFile(fileObj, progress);
        }
    } // export

    /**
     * Copies a file to the Web export directory.
     * @param progress - Export progress holder object that holds the file
     * object to copy and is used to display export status information
     * @param bForce - <tt>True</tt> to force copy even if the exported file
     * already exists and is younger than the original
     * @throws ExportException on failure
     */
    public void copy
        ( ExportProgress progress
        , boolean        bForce
        )
        throws ExportException
    {
        FileObject fileObj = progress.getFileObject();
        File       dir, srcFile, destFile;
        long       expMod;
        String     name;

        // Ensure that the output directory exists.
        dir = this.prepareOutputDir(progress);
        // Prepare the progrss object.
        name = fileObj.getFileName().toLowerCase();
        progress.setMainFileName(name);
        srcFile = new File(fileObj.getFullPath());
        destFile = new File(dir, name);
        if (destFile.exists())
            expMod = destFile.lastModified();
        else
            expMod = 0;
        if (bForce || srcFile.lastModified() > expMod)
            progress.setMainUpToDate(false);
        else
        {
            progress.setMainUpToDate(true);
            return;
        }
        // Copy the file.
        FileInputStream in = null;
        FileOutputStream out = null;
        byte[] buf = new byte[32 * 1024];
        int len;
        try
        {
            in = new FileInputStream(srcFile);
            out = new FileOutputStream(destFile);
            while ((len = in.read(buf)) > 0)
                out.write(buf, 0, len);
        }
        catch (Exception e)
        {
            throw new ExportException(fileObj, "Cannot copy file \"" +
                srcFile.toString() + "\" to \"" + destFile.toString() + "\"");
        }
        finally
        {
            try
            {
                if (in != null)
                    in.close();
                if (out != null)
                    out.close();
            }
            catch (Exception exc)
            {
            }
        }
    } // copy

    /**
     * Gets the exported files for a file if they exist (null otherwise).
     * @param fsObj - File system object whose exported files are needed
     * @return The exported files (normal and small) or <tt>null</tt> if
     * non-existent
     */
    public File[] getExportedFiles (AbstractFSObject fsObj)
    {
        DirectoryObject parentDir, seriesDir;
        File            dir, f;
        String          name;
        File[]          str = new File[2];

        str[0] = str[1] = null;
        // Get the output directory.
        parentDir = seriesDir = (DirectoryObject) fsObj.getParent();
        if (parentDir == null)
            return str;
        while (seriesDir.getParent() != null)
            seriesDir = (DirectoryObject) seriesDir.getParent();
        dir = new File(m_settings.get(Settings.EXPORT_DIRECTORY),
            parentDir.getPath(true).toLowerCase());
        // Delete the files.
        if (fsObj instanceof DirectoryObject)
            return str;
        else if (fsObj instanceof FileObject)
        {
            // Delete the two JPEG files.
            name = ((FileObject) fsObj).getBaseFileName().toLowerCase();
            f = new File(dir, name + ".jpg");
            if (f.exists())
                str[0] = f;
            f = new File(dir, name + "-s.jpg");
            if (f.exists())
                str[1] = f;
        }
        return str;
    } // getExportedFiles

    /**
     * Renames the exported files when a file object is about to be renamed.
     * @param oldFsObj - Old file still having its old name
     * @param newParent - New parent directory or <tt>null</tt> if the parent
     * has not changed
     * @param newName - New name or <tt>null</tt> if the name has not changed
     * @throws ExportException
     */
    public void renameExportedFiles
        ( AbstractFSObject oldFsObj
        , DirectoryObject  newParent
        , String           newName
        ) throws ExportException
    {
        DirectoryObject newParentDirObj;
        File            exportDir, oldDir, newPDir, newDir, oldFile, newFile;
        String          oldBaseName, newBaseName;

        // Prepare the name to the new parent.
        exportDir = new File(m_settings.get(Settings.EXPORT_DIRECTORY));
        newParentDirObj = (newParent != null) ?
            newParent : (DirectoryObject) oldFsObj.getParent();
        if (newParentDirObj == null)
            newPDir = exportDir;
        else
        {
            newPDir = new File(exportDir, newParentDirObj.getPath(true).
                toLowerCase());
        }
        if (oldFsObj instanceof DirectoryObject)
        {
            // Need to rename the directory.
            if (newName != null)
                newName = newName.toLowerCase();
            oldDir = new File(exportDir, oldFsObj.getPath(true).toLowerCase());
            newDir = new File(newPDir, (newName != null) ? newName :
                oldFsObj.getFileName().toLowerCase());
            if (!oldDir.renameTo(newDir))
            {
                System.err.println("W: Cannot rename exported directory from:"
                    + "\n   " + oldDir.getAbsolutePath() + "\nto:\n   "
                    + newDir.getAbsolutePath());
            }
            return;
        }
        // Rename the exported files (main and index).
        if (!(oldFsObj instanceof FileObject))
            return;
        FileObject fileObj = (FileObject) oldFsObj;
        if (oldFsObj.getParent() == null)
            oldDir = exportDir;
        else
        {
            oldDir = new File(exportDir,
                oldFsObj.getParent().getPath(true).toLowerCase());
        }
        oldBaseName = fileObj.getBaseFileName();
        newBaseName = (newName != null) ? newName : fileObj.getBaseFileName();
        int i = newBaseName.indexOf(".");
        if (i > 0)
            newBaseName = newBaseName.substring(0, i);
        // Rename the main file.
        oldFile = new File(oldDir, oldBaseName + ".jpg");
        newFile = new File(newPDir, newBaseName + ".jpg");
        if (!oldFile.renameTo(newFile))
        {
            System.err.println("W: Cannot rename exported main file from:\n   "
                + oldFile.getAbsolutePath() + "\nto:\n   "
                + newFile.getAbsolutePath());
        }
        // Rename the index file.
        oldFile = new File(oldDir, oldBaseName + "-s.jpg");
        newFile = new File(newPDir, newBaseName + "-s.jpg");
        if (!oldFile.renameTo(newFile))
        {
            System.err.println("W: Cannot rename exported index file from:\n   "
                + oldFile.getAbsolutePath() + "\nto:\n   "
                + newFile.getAbsolutePath());
        }
    } // renameExportedFiles

    /**
     * Deletes the exported files for a file system object from the Web export
     * directory.
     * @param fsObj - File object whose exported files should be deleted
     * @param whatTodelete - Can be one of <tt>EXPORT_ALL</tt>, delete all
     * image forms; <tt>EXPORT_MAIN_IMAGE</tt>, delete only the main image; or
     * <tt>EXPORT_OVERVIEW_IMAGE</tt>, delete only the overview image.
     * @throws ExportException
     */
    public void deleteExportedFiles
        ( AbstractFSObject fsObj
        , int              whatTodelete
        )
        throws ExportException
    {
        DirectoryObject parentDir, seriesDir;
        File            dir, f;
        String          name;

        // Get the output directory exists.
        parentDir = seriesDir = (DirectoryObject) fsObj.getParent();
        if (parentDir == null)
            return;
        while (seriesDir.getParent() != null)
            seriesDir = (DirectoryObject) seriesDir.getParent();
        dir = new File(m_settings.get(Settings.EXPORT_DIRECTORY),
            parentDir.getPath(true).toLowerCase());
        if (!dir.isDirectory())
            return;
        // Delete the files.
        if (fsObj instanceof DirectoryObject)
        {
            // Delete all exported JPEG files.
            // ...
        }
        else if (fsObj instanceof FileObject)
        {
            // Delete the two JPEG files.
            name = ((FileObject) fsObj).getBaseFileName().toLowerCase();
            if (whatTodelete == IWebExport.EXPORT_ALL ||
                whatTodelete == IWebExport.EXPORT_MAIN_IMAGE)
            {
                f = new File(dir, name + ".jpg");
                f.delete();
            }
            if (whatTodelete == IWebExport.EXPORT_ALL ||
                whatTodelete == IWebExport.EXPORT_OVERVIEW_IMAGE)
            {
                f = new File(dir, name + "-s.jpg");
                f.delete();
            }
        }
    } // deleteExportedFiles
} // BaseExporter
