/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ExportThread
    Created: 10 January, 2003
        $Id: ExportThread.java 158 2009-05-06 19:49:13Z dirk $
  $Revision: 158 $
      $Date: 2009-05-06 21:49:13 +0200 (Mi, 06 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.webexport;

import java.util.Iterator;
import java.util.List;

import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.ui.ExplorerMainFrame;

/**
 * Thread class that executes the Web export.
 */
public class ExportThread extends Thread implements Runnable
{
    private Settings          m_settings;
    private SeriesContainer   m_seriesContainer;
    private ExplorerMainFrame m_frame;

    private List<FileObject>  m_exportList;
    private int               m_whatToExport;
    private boolean           m_bForce;

    /**
     * Constructor of a new <tt>ExportThread</tt> instance.
     * @param exportList - List of objects to be exported
     * @param settings - Settings object
     * @param sContainer - Series container object
     * @param frm - Parent main frame in the user interface
     */
    public ExportThread
        ( List<FileObject>  exportList
        , Settings          settings
        , SeriesContainer   sContainer
        , ExplorerMainFrame frm
        )
    {
        m_exportList = exportList;
        m_settings = settings;
        m_seriesContainer = sContainer;
        m_frame = frm;
        m_whatToExport = IWebExport.EXPORT_ALL;
        m_bForce = false;
    } // ExportThread

    /**
     * Sets the export options.
     * @param bMain
     * @param bIndex
     * @param bForce
     */
    public void setExportOptions
        ( boolean bMain
        , boolean bIndex
        , boolean bForce
        )
    {
        if (bMain && bIndex)
            m_whatToExport = IWebExport.EXPORT_ALL;
        else if (bMain)
            m_whatToExport = IWebExport.EXPORT_MAIN_IMAGE;
        else if (bIndex)
            m_whatToExport = IWebExport.EXPORT_OVERVIEW_IMAGE;
        else
            m_whatToExport = IWebExport.EXPORT_NOTHING;
        m_bForce = bForce;
    } // setExportOptions

    /** Runs to execute the Web export. */
    @Override
	public void run ()
    {
        Iterator<FileObject> iter;
        FileObject           fileObj;
        IWebExport           exp;
        ExportProgress       progress = null;
        int                  currentFileNumber, outOfMemoryCount;

        exp = ExportFactory.getExport(m_settings, m_seriesContainer, m_frame);
        m_frame.exportSetReady(false);
        try
        {
            iter = m_exportList.iterator();
            currentFileNumber = 0;
            while (iter.hasNext())
            {
                fileObj = iter.next();
                currentFileNumber++;
                outOfMemoryCount = 0;
                while (fileObj != null)
                {
                    try
                    {
                        progress = new ExportProgress
                            (fileObj, currentFileNumber);
                        m_frame.setExportProgress(progress);
                        if (fileObj.getFileType() ==
                            FileObject.TYPE_IMAGE_PREVIEW)
                        {
                            exp.export(progress, m_whatToExport, m_bForce);
                        }
                        else
                            exp.copy(progress, m_bForce);
                        fileObj = null;
                    }
                    catch (OutOfMemoryError me)
                    {
                        outOfMemoryCount++;
                        if (outOfMemoryCount > 4)
                        {
                            // Give up exporting.
                            throw new ExportException
                                (fileObj, "Out of memory");
                        }
                        // Try again later.
                        try
                        {
                            Thread.sleep(333 + outOfMemoryCount * 555);
                        }
                        catch (InterruptedException ie)
                        {
                        }
                        // We process the item again.
                        Runtime.getRuntime().gc();
                    }
                }
                if (m_frame.m_exportStop)
                {
                    m_frame.exportSetReady(true);
                    m_frame.setExportProgress(null);
                    return;
                }
            }
        }
        catch (Exception e)
        {
            System.err.println("E: Export (thread) exception:\n" + e);
            e.printStackTrace();
            m_frame.exportSetError(e.getMessage());
        }
        m_frame.exportSetReady(true);
        m_frame.setExportProgress(null);
    } // run
} // ExportThread
